/**
 * i-net software provides programming examples for illustration only, without warranty
 * either expressed or implied, including, but not limited to, the implied warranties
 * of merchantability and/or fitness for a particular purpose. This programming example
 * assumes that you are familiar with the programming language being demonstrated and
 * the tools used to create and debug procedures. i-net software support professionals
 * can help explain the functionality of a particular procedure, but they will not modify
 * these examples to provide added functionality or construct procedures to meet your
 * specific needs.
 *
 * Copyright © 1999-2025 i-net software GmbH, Berlin, Germany.
**/
package viewer;

import java.awt.Dimension;
import java.awt.event.WindowEvent;
import java.awt.event.WindowFocusListener;
import java.io.IOException;
import java.net.URISyntaxException;
import java.util.HashMap;
import java.util.Map;

import javax.swing.JFrame;

import com.inet.report.EngineRenderData;
import com.inet.report.ReportException;
import com.inet.viewer.ReportView;
import com.inet.viewer.SwingReportViewer;

/**
 * A simple example showing how to show multiple reports not in tabs in the viewer, but how to have each report be in
 * it's own frame.
 */
public class ViewerWithMultipleFrames {

    private final SwingReportViewer viewer;

    /**
     * Displays multiple reports, each report in it's own frame.
     */
    public ViewerWithMultipleFrames() {

        viewer = new SwingReportViewer() {
            // Here we will override the default behavior of the Viewer.
            // For this, we'll keep track of the various Views in a HashMap:

            private Map<JFrame, ReportView> myMap = new HashMap<JFrame, ReportView>();

            @Override
            public void addReportView( ReportView view, boolean closeable ) {

                // We now override the default behavior of openReportView
                // which was to open new tabs for each new view.
                // Instead, we open a new frame, place the new report view
                // inside it, and remember it in combination with the frame
                // in our hash map.

                JFrame repFrame = new JFrame( view.getReportData().getReportTitle() );

                // So that the viewer knows which ReportView is the
                // current report view (for toolbar actions, etc.), we'll
                // define a WindowFocusListener so that whenever a user
                // focuses on a report view window, it becomes the
                // current report view.

                WindowFocusListener l = new WindowFocusListener() {
                    @Override
                    public void windowGainedFocus( WindowEvent e ) {
                        setCurrentReportView( myMap.get( e.getWindow() ) );
                    }

                    @Override
                    public void windowLostFocus( WindowEvent e ) {
                        // Nothing to do
                    }
                };
                myMap.put( repFrame, view );

                // Now we simply add the view to the frame and show it:

                repFrame.addWindowFocusListener( l );
                repFrame.getContentPane().add( view.getComponent() );
                repFrame.setDefaultCloseOperation( JFrame.EXIT_ON_CLOSE );
                repFrame.pack();
                repFrame.setVisible( true );
                setCurrentReportView( view );
            }

        };

        String reportLocation = "";
        try {
            reportLocation = getClass().getResource( "../sample.rpt" ).toURI().toString();
        } catch( URISyntaxException e ) {
            // Nothing to do
        }

        // We're just using an engine render data for now
        EngineRenderData renderData = new EngineRenderData( "report=" + reportLocation );
        viewer.addNewReportView( renderData );
        viewer.addNewReportView( renderData );

        // If you have a server running, you can use the URL: http://server:port/report.rpt
        // viewer.addNewReportView( new URLRenderData( "http://server:port?report=" + loc ) );

        viewer.setPreferredSize( new Dimension( 725, 35 ) );
    }

    /**
     * Initialize the GUI.
     */
    public static void initGUI() {
        ViewerWithMultipleFrames myApp = new ViewerWithMultipleFrames();
        JFrame frame = new JFrame();
        frame.setDefaultCloseOperation( JFrame.EXIT_ON_CLOSE );
        frame.getContentPane().add( myApp.viewer );
        //Display the window.
        frame.pack();
        frame.setVisible( true );
    }

    /**
     * Main method of this sample
     * @param args arguments not used
     * @throws IOException in case of IO errors. Often the port is already in use.
     * @throws ReportException a Report exception
     */
    public static void main( String[] args ) throws IOException, ReportException {
        initGUI();
    }
}
